"use strict"

var editMode;

// takes array of arrays of ints, outer dimension is Z, inner is X
// returns object with coords and colors Float32Arrays,
// each having 3 floats per vertex for drawing as a triangles array
function heightmapToModel(heightmap)
{
 var model={}
 setCoordsAndColors(model,heightmap);
 return model
}

function refreshModelFromHeightmap(model,heightmap)
{
 setCoordsAndColors(model,heightmap);
 if(model.context==context)
 {  
  context.bindBuffer(context.ARRAY_BUFFER,model.aCoord);
  context.bufferData(context.ARRAY_BUFFER,model.coords,context.STATIC_DRAW); 
  context.bindBuffer(context.ARRAY_BUFFER,model.aColor);
  context.bufferData(context.ARRAY_BUFFER,model.colors,context.STATIC_DRAW);
 }
 model.numCoordFloats=model.coords.length;
}

function setCoordsAndColors(model,heightmap)
{
 model.coords=[]
 model.colors=[]
 for(var z=0;z<heightmap.length;++z)
 {
  var row=heightmap[z];
  for(var x=0;x<row.length;++x)
  {
   var y=row[x]|0;
   var west=(x==0)?-1:row[x-1];
   var east=(x==row.length-1)?-1:row[x+1];
   var north=(z==0)?-1:heightmap[z-1][x];
   var south=(z==heightmap.length-1)?-1:heightmap[z+1][x];
   if(y>=0) { addQuad(model,x,y,z,"u"); }
   for(var y2=west;y2<y;++y2) { addQuad(model,x,y2,z,"w"); }
   for(var y2=east;y2<y;++y2) { addQuad(model,x,y2,z,"e"); }
   for(var y2=north;y2<y;++y2) { addQuad(model,x,y2,z,"n"); }
   for(var y2=south;y2<y;++y2) { addQuad(model,x,y2,z,"s"); }   
  }
 }
 model.coords=new Float32Array(model.coords);
 model.colors=new Float32Array(model.colors); 
}

const PLAYER_WIDTH=1/2;
const PLAYER_HEIGHT=9/8;
const ITEM_SIZE=1/4;
const CHECKPOINT_SIZE=1/8;

const CUBE_COORDS=[
 0,1,0, 0,1,1, 1,1,0,  1,1,1, 1,1,0, 0,1,1,
 0,0,1, 1,0,1, 0,1,1,  1,1,1, 0,1,1, 1,0,1,
 1,0,0, 1,1,0, 1,0,1,  1,1,1, 1,0,1, 1,1,0,
 0,0,0, 0,1,0, 1,0,0,  1,1,0, 1,0,0, 0,1,0,
 0,0,0, 0,0,1, 0,1,0,  0,1,1, 0,1,0, 0,0,1,
 0,0,0, 1,0,0, 0,0,1,  1,0,1, 0,0,1, 1,0,0, 
]

var skyboxModel;

var unscaledPlayerModel={
 "coords":  CUBE_COORDS,
 "colors":[
  1,1,1/2, 1,1,1/2, 1,1,1/2, 1,1,1/2, 1,1,1/2, 1,1,1/2,
  1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4,
  1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8,
  1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4, 1/2,1/2,1/4,
  1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8, 1/4,1/4,1/8,
  0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0,
 ]
}

var unscaledItemModel={
 "coords":  CUBE_COORDS,
 "colors":[
  1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,  
  1,0,0, 1,0,0, 1,0,0, 1,0,0, 1,0,0, 1,0,0,
  0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 0,1,0, 
  0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 0,1,1, 
  1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1, 1,0,1,
  0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 0,0,1, 
 ]
}

var unscaledCheckpointModel={
 "coords":  CUBE_COORDS,
 "colors":[
  1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 
  0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 
  0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0,
  0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 0.5,0.5,0, 
  0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0, 0,0,0,
  1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0, 1,1,0,  
 ]
}



const QUAD_COLORS={
 "u0":[1/4,3/4,1/4],
 "u1":[1/4,1/2,1/4],
 "s1":[1/8,3/8,1/8],
 "s0":[1/8,1/4,1/8],
 "e1":[3/16,9/16,3/16],
 "e0":[3/16,3/8,3/16],
 "w1":[3/16,9/16,3/16],
 "w0":[3/16,3/8,3/16],
 "n1":[1/8,3/8,1/8],
 "n0":[1/8,1/4,1/8], 
}

const QUAD_COORDS={
 "u":[0,0,0, 0,0,1, 1,0,0, 1,0,1, 1,0,0, 0,0,1],
 "n":[0,0,0, 0,1,0, 1,0,0, 1,1,0, 1,0,0, 0,1,0],
 "s":[1,1,1, 0,1,1, 1,0,1, 0,0,1, 1,0,1, 0,1,1],
 "w":[0,0,0, 0,0,1, 0,1,0, 0,1,1, 0,1,0, 0,0,1],
 "e":[1,1,1, 1,0,1, 1,1,0, 1,0,0, 1,1,0, 1,0,1], 
}

const QUAD_VALUES={
 "u":1,
 "n":3/4,
 "s":3/4,
 "e":1/2,
 "w":1/2
}

const RGB_TABLE=[


 [0,1,0],
 [1,1,0],
 [1,0,0], 
 [1,0,1],
 [0,0,1],
 [0,1,1],
 

]

function addQuad(model,x,y,z,dir)		 
{
 var parity=((x+z)&1)==worldColorParity;
 
 var blockLightness=parity?1:15/16;
 var lightness=blockLightness*QUAD_VALUES[dir];
 var rgb=RGB_TABLE[(y+(dir=="u"?0:1))%6]
 var saturation=parity?2/16:3/16;
 if(y>5) { saturation-=3/64; }


 
 var red=lightness*(rgb[0]*saturation+1-saturation);
 var green=lightness*(rgb[1]*saturation+1-saturation);
 var blue=lightness*(rgb[2]*saturation+1-saturation);

 
 var coords=QUAD_COORDS[dir];
 for(var i=0;i<coords.length;i+=3)
 {
  model.coords.push(coords[i+0]+x);
  model.coords.push(coords[i+1]+y);
  model.coords.push(coords[i+2]+z);
  model.colors.push(red);
  model.colors.push(green);
  model.colors.push(blue);
 }
}

var worldHeightmap;
var worldColorParity; // to allow odd-size insertions
var canvas,context;
var worldModel;
var playerModel;
var itemModel, checkpointModel;
var player;
var spawnPosition, spawnFacingNumber;
var items,itemsForReset, checkpoints, currentCheckpoint, checkpointFacing;
var cameraTheta, cameraRawPhi; // in radians and [0,1] respectively
var aCoord,aColor;
var aCoordDynamic, aColorDynamic;
var uPerspective, uCameraPosition, uCameraAngle, uObjectPosition,uAlpha;
var uObjectRotation;
var frameNumber,drawNumber,itemsGotten, victoryFrame;
var gameLogicZeroOffset;
var mainProgram,skyProgram;
var checkpointFrame, frameNumberTicking;
var overlayCommand;

var uSkyPerspective,uSkyCameraAngle,aSkyCoord;

var launched;

function onBodyLoad()
{
 document.getElementById("buttonS").onmousedown=tameButton;
 document.getElementById("buttonR").onmousedown=tameButton;
 document.getElementById("buttonM").onmousedown=tameButton;
 
 launched=false;
 var h=window.location.hash;
 if(h && h[0]=="#") { h=h.slice(1); }
 var w=decodeWorld(h); 
 if(w)
 {
  document.getElementById("play_button").onclick=function()
  { launchGame(h,false); }
  document.getElementById("edit_button").onclick=function()
  { launchGame(h,true); }
  document.getElementById("starttext").innerHTML=(
   "WELCOME TO RAINBOW BULLSHIT LAND v2!<br>"+
   "The current map has "+w.items.length+" RAINBOW BULLSHIT."
  )
 }
 else
 {
  if(h)
  {
   document.getElementById("starttext").innerHTML=(
    "WELCOME TO RAINBOW BULLSHIT LAND v2!<br>"+
    "The map in the URL string failed to load.<br>You can edit a blank map."
   )
  }
  else
  {
   document.getElementById("starttext").innerHTML=(
    "WELCOME TO RAINBOW BULLSHIT LAND v2!<br>"+
    "There is no map loaded.<br>You can edit a blank map."
   )   
  }
  document.getElementById("play_button").style.display="none";
  document.getElementById("edit_button").onclick=function()
  { launchGame(false,true); }  
 }
 
 document.getElementById("startmenu").style.display="block"; 
}

function launchGame(savedString,enableEdit)
{
 if(launched) { return; }
 document.getElementById("startmenu").style.display="none";
 editMode=enableEdit;
 keysHeld={}
 mouseHeld={}
 keysFresh={}
 mouseAccumulator=[0,0];
 window.onkeydown=onKeyDown;
 window.onkeyup=onKeyUp;
 window.onmousedown=onMouseDown;
 window.onmouseup=onMouseUp; 
 window.onmousemove=onMouseMove;
 document.onpointerlockchange=onPointerLockChange;
 var savedWorld=decodeWorld(savedString);
 if(savedWorld)
 {
  worldHeightmap=savedWorld.heightmap;
  worldColorParity=savedWorld.colorParity;
 }
 else { worldHeightmap=makeEmptyHeightmap(8,8,0); worldColorParity=0; }
 worldModel=heightmapToModel(worldHeightmap);
 makePlayerModel();
 makeItemModel();
 makeCheckpointModel(); 
 makeSkyboxModel();
 initContext();
 if(savedWorld)
 {
  spawnPosition=savedWorld.spawnPosition;
  spawnFacingNumber=savedWorld.spawnFacing;
 }
 else
 {
  spawnPosition=[
   3.5-PLAYER_WIDTH/2,
   0,
   3.5-PLAYER_WIDTH/2
  ];
  spawnFacingNumber=0;
 }
 player={width:PLAYER_WIDTH,
	 height:PLAYER_HEIGHT}
 respawnPlayer();
 if(savedWorld)
 {
  items=savedWorld.items;
  checkpoints=savedWorld.checkpoints;
 }
 else
 {
  items=[];
  checkpoints=[];
  history.replaceState("","","#"+encodeWorld())
 }
 currentCheckpoint=null;
 itemsForReset=JSON.stringify(items);

 cameraRawPhi=0.5;
 victoryFrame=0;
 frameNumber=0;
 frameNumberTicking=enableEdit;
 drawNumber=0;
 itemsGotten=0;
 overlayCommand=null;

 updateText();
 canvas.requestPointerLock();
 drawEverything();
 gameLogicZeroOffset=performance.now();
 setInterval(gameLogicFrameCheck,1000/240);
}

function stripSpaces(str)
{
 var chars=[]
 for(var i=0;i<str.length;++i)
 {
  var c=str[i];
  if(c=="." || (c>="A" && c<="Z") || (c>="a" && c<="z")) { chars.push(c); }
 }
 return chars;
}

function makePlayerModel()
{
 playerModel={coords:unscaledPlayerModel.coords.slice(),
	      colors:unscaledPlayerModel.colors.slice()}
 for(var i=0;i<playerModel.coords.length;++i)
 {
  playerModel.coords[i]*=(i%3==1)?PLAYER_HEIGHT:PLAYER_WIDTH;
 }
 playerModel.colors=new Float32Array(playerModel.colors);
 playerModel.coords=new Float32Array(playerModel.coords); 
}

function makeSkyboxModel()
{
 skyboxModel={coords:[]};
 for(var i=0;i<CUBE_COORDS.length;++i)
 {
  skyboxModel.coords.push(CUBE_COORDS[i]*2-1);
 }
 skyboxModel.coords=new Float32Array(skyboxModel.coords);
}

function makeItemModel()
{
 itemModel={coords:unscaledItemModel.coords.slice(),
	      colors:unscaledItemModel.colors.slice()}
 for(var i=0;i<itemModel.coords.length;++i)
 {
  itemModel.coords[i]*=ITEM_SIZE;
  itemModel.coords[i]-=ITEM_SIZE/2;
 }
 itemModel.colors=new Float32Array(itemModel.colors);
 itemModel.coords=new Float32Array(itemModel.coords); 
}

function makeCheckpointModel()
{
 checkpointModel={coords:unscaledCheckpointModel.coords.slice(),
	      colors:unscaledCheckpointModel.colors.slice()}
 for(var i=0;i<checkpointModel.coords.length;++i)
 {
  checkpointModel.coords[i]*=CHECKPOINT_SIZE;
  checkpointModel.coords[i]-=CHECKPOINT_SIZE/2;
 }
 checkpointModel.colors=new Float32Array(checkpointModel.colors);
 checkpointModel.coords=new Float32Array(checkpointModel.coords); 
}

function initContext()
{
 canvas=document.getElementById("canvas");
 canvas.style.display="block";
 context=canvas.getContext("webgl");


 var vShader=context.createShader(context.VERTEX_SHADER);
 var fShader=context.createShader(context.FRAGMENT_SHADER);
 context.shaderSource(vShader,VERTEX_SHADER_SOURCE)
 context.compileShader(vShader)
 context.shaderSource(fShader,FRAGMENT_SHADER_SOURCE)
 var log=context.getShaderInfoLog(vShader);
 if(log) { console.log(log); }
 context.compileShader(fShader);
 log=context.getShaderInfoLog(fShader);
 if(log) { console.log(log); }
 mainProgram=context.createProgram(); 
 context.attachShader(mainProgram,vShader);
 context.attachShader(mainProgram,fShader);
 context.linkProgram(mainProgram);
 context.useProgram(mainProgram);

 
 var vSkyShader=context.createShader(context.VERTEX_SHADER);
 var fSkyShader=context.createShader(context.FRAGMENT_SHADER);
 context.shaderSource(vSkyShader,SKY_VERTEX_SHADER_SOURCE)
 context.compileShader(vSkyShader)
 context.shaderSource(fSkyShader,SKY_FRAGMENT_SHADER_SOURCE)
 var log=context.getShaderInfoLog(vSkyShader);
 if(log) { console.log(log); }
 context.compileShader(fSkyShader);
 log=context.getShaderInfoLog(fSkyShader);
 if(log) { console.log(log); }
 skyProgram=context.createProgram(); 
 context.attachShader(skyProgram,vSkyShader);
 context.attachShader(skyProgram,fSkyShader);
 context.linkProgram(skyProgram);
 

 uPerspective=context.getUniformLocation(mainProgram,"uPerspective");
 uObjectPosition=context.getUniformLocation(mainProgram,"uObjectPosition");
 uObjectRotation=context.getUniformLocation(mainProgram,"uObjectRotation"); 
 uCameraPosition=context.getUniformLocation(mainProgram,"uCameraPosition");
 uCameraAngle=context.getUniformLocation(mainProgram,"uCameraAngle");
 uAlpha=context.getUniformLocation(mainProgram,"uAlpha");  
 aCoord=context.getAttribLocation(mainProgram,"aCoord");
 aColor=context.getAttribLocation(mainProgram,"aColor");
 
 context.enable(context.CULL_FACE);
 context.blendFuncSeparate(context.SRC_ALPHA,context.ONE_MINUS_SRC_ALPHA,
			   context.ZERO,context.ONE); 
 context.clearColor(0,0,0,1);
 
 aCoordDynamic=context.createBuffer();
 aColorDynamic=context.createBuffer();
 
 

 uSkyPerspective=context.getUniformLocation(skyProgram,"uSkyPerspective");
 uSkyCameraAngle=context.getUniformLocation(skyProgram,"uSkyCameraAngle");
 aSkyCoord=context.getAttribLocation(skyProgram,"aSkyCoord"); 
}

function initVBOs(model)
{
 model.context=context;
 model.aCoord=context.createBuffer();
 context.bindBuffer(context.ARRAY_BUFFER,model.aCoord);
 context.bufferData(context.ARRAY_BUFFER,model.coords,context.STATIC_DRAW);
 model.aColor=context.createBuffer();
 context.bindBuffer(context.ARRAY_BUFFER,model.aColor);
 context.bufferData(context.ARRAY_BUFFER,model.colors,context.STATIC_DRAW);
 model.numCoordFloats=model.coords.length;
}

function initSkyVBOs(model)
{
 model.context=context;
 model.aSkyCoord=context.createBuffer();
 context.bindBuffer(context.ARRAY_BUFFER,model.aSkyCoord);
 context.bufferData(context.ARRAY_BUFFER,model.coords,context.STATIC_DRAW);
 model.numCoordFloats=model.coords.length; 
}

const CAMERA_PHI_RATIO=Math.PI/2;
const CAMERA_THETA_RATIO=Math.PI;
const CAMERA_MAX_DISTANCE=48;
const CAMERA_START_DISTANCE=24;
const CAMERA_MIN_DISTANCE=8;
const CAMERA_DISTANCE_RATIO=64;

const IDENTITY_ROTATION=new Float32Array([1,0,0,0,1,0,0,0,1]);

function drawEverything()
{
 if(document.pointerLockElement==canvas) {++drawNumber;}
 if(context.isContextLost())
 {
  initContext();
 }
 var widthWanted=Math.max(64,Math.floor(canvas.clientWidth))
 if(canvas.width!=widthWanted) { canvas.width=widthWanted; }
 var heightWanted=Math.max(64,Math.floor(canvas.clientHeight))
 if(canvas.height!=heightWanted) { canvas.height=heightWanted; }
 
 context.viewport(0,0,canvas.width,canvas.height);
 context.clear(context.COLOR_BUFFER_BIT|context.DEPTH_BUFFER_BIT);

 var phi0=Math.PI*(cameraRawPhi*3/8-3/16);
 var phi1=Math.PI*(cameraRawPhi*3/8+1/8);
 
 var cosP0=Math.cos(phi0);
 var sinP0=Math.sin(phi0);
 var cosP1=Math.cos(phi1);
 var sinP1=Math.sin(phi1); 
 var cosT=Math.cos(cameraTheta);
 var sinT=Math.sin(cameraTheta);  


 var camAngle0=new Float32Array([
  cosT,sinT*sinP0,-sinT*cosP0,
  0,cosP0,sinP0,
  sinT,-cosT*sinP0,cosT*cosP0  
 ]);
 var camPosition0=[
   player.position[0]+PLAYER_WIDTH/2,
   player.position[1]+PLAYER_HEIGHT+1/4,
   player.position[2]+PLAYER_WIDTH/2,
  ]

 var camAngle1=new Float32Array([
  cosT,sinT*sinP1,-sinT*cosP1,
  0,cosP1,sinP1,
  sinT,-cosT*sinP1,cosT*cosP1
 ]);
 var camPosition1=[
  player.position[0]+PLAYER_WIDTH/2-sinT*cosP1*12,
  5+sinP1*12,
  player.position[2]+PLAYER_WIDTH/2+cosT*cosP1*12,
 ]

 var camAngle2=new Float32Array([
  cosT,sinT,0,
  0,0,1,
  sinT,-cosT,0,
 ]);
 var camPosition2=[
  player.position[0]+PLAYER_WIDTH/2,
  16+cameraRawPhi*48,
  player.position[2]+PLAYER_WIDTH/2,
 ]  

 // angled third-person view:
 // left 1/3 of display, minus 8-px margin on all sides
 var xGuide=Math.ceil(canvas.width*2/3);
 drawView(8,8,xGuide-16,canvas.height-16,
	  camPosition1,camAngle1,10/3,true,true); 

 // subviews: right 1/3 of display, minus 8-px margin on right
 // first-person is bottom half, minus 4-px above and 8 below
 // distant is top half, minus 8-px above and 4 below
 var yGuide=Math.ceil(canvas.height/2);
 drawView(xGuide,8,canvas.width-xGuide-8,yGuide-12,
	  camPosition0,camAngle0,10/3,false,true);

 drawView(xGuide,yGuide+4,canvas.width-xGuide-8,canvas.height-yGuide-12,
	  camPosition2,camAngle2,16/3,true,false);
 
 requestAnimationFrame(drawEverything);
}
 
function drawView(viewLeft,viewBottom,viewWidth,viewHeight,
		  camPosition,camAngle,zoom,doSeethrough,doSky)
{
 var perspMatrix;
 var a=viewHeight/viewWidth;


 
 context.viewport(viewLeft,viewBottom,viewWidth,viewHeight);
 perspMatrix=makePerspectiveMatrix(zoom*a,zoom,1/64,640,0);

 if(doSky)
 {
  context.useProgram(skyProgram);
  context.enableVertexAttribArray(aSkyCoord);
 
  context.uniformMatrix4fv(uSkyPerspective,false,perspMatrix);
  context.uniformMatrix3fv(uSkyCameraAngle,false,camAngle);
  context.disable(context.CULL_FACE);
  context.disable(context.DEPTH_TEST);
  context.disable(context.BLEND);
  drawSkyModel();
  context.disableVertexAttribArray(aSkyCoord);
 }
 
 context.useProgram(mainProgram);
 context.enableVertexAttribArray(aCoord);
 context.enableVertexAttribArray(aColor);
 context.uniformMatrix4fv(uPerspective,false,perspMatrix);
 context.enable(context.CULL_FACE);

 var cos=Math.cos(frameNumber%128*Math.PI/64)
 var sin=Math.sin(frameNumber%128*Math.PI/64)
 var spin=new Float32Array([
  cos,0,sin,
  0,1,0,
  -sin,0,cos
 ]);

 var spin=new Float32Array([
  cos,sin*sin,-sin*cos,
  0,cos,sin,
  sin,-cos*sin,cos*cos
 ]);

 

 
// console.log(cameraPosition);

 context.uniform3fv(uCameraPosition,new Float32Array(camPosition));
 
 context.uniformMatrix3fv(uCameraAngle,false,camAngle);
 context.clear(context.DEPTH_BUFFER_BIT);
 context.enable(context.DEPTH_TEST);
 context.disable(context.BLEND); 
 drawModel(worldModel,[0,0,0],1,IDENTITY_ROTATION);
 drawModel(playerModel,player.position,1,IDENTITY_ROTATION);
 for(var i=0;i<items.length;++i)
 {
  drawModel(itemModel,items[i].drawPosition,1,spin);
 }
 for(var i=0;i<checkpoints.length;++i)
 {
  if(checkpoints[i]!==currentCheckpoint)
  {
   drawModel(checkpointModel,checkpoints[i].drawPosition,1,
	     IDENTITY_ROTATION
	    );
  }
 }
 

 context.enable(context.BLEND);
 var shadowCoords=computeShadowCoords();
 if(shadowCoords.length)
 {
  perspMatrix=makePerspectiveMatrix(zoom*a,zoom,1/64,640,
				    -1/640/256);
   context.uniformMatrix4fv(uPerspective,false,perspMatrix);  
  var shadowColors=new Array();
  for(var i=0;i<shadowCoords.length;++i)
  {
   shadowColors.push(editMode?1/2:0);
  }
  drawDynamic(shadowCoords,
	      shadowColors,
	      [0,0,0],
	      editMode?6/32:3/32,
	      IDENTITY_ROTATION);
  perspMatrix=makePerspectiveMatrix(zoom*a,zoom,1/64,640,0);
  context.uniformMatrix4fv(uPerspective,false,perspMatrix);    
 }

 if(doSeethrough)
 {
  context.clear(context.DEPTH_BUFFER_BIT);
  drawModel(playerModel,player.position,1/8,IDENTITY_ROTATION);
 
  for(var i=0;i<items.length;++i)
  {
   drawModel(itemModel,items[i].drawPosition,1/8,spin);
  }
  for(var i=0;i<checkpoints.length;++i)
  {
   if(checkpoints[i]!==currentCheckpoint)
   {
    drawModel(checkpointModel,checkpoints[i].drawPosition,1/8,
	      IDENTITY_ROTATION
	     );
   }
  }  
 }

 if(editMode && drawNumber%30<29)
 {
  drawModel(playerModel,spawnPosition,1/2,IDENTITY_ROTATION);  
 }

 context.disableVertexAttribArray(aCoord);
 context.disableVertexAttribArray(aColor); 

}


function makePerspectiveMatrix(xScale,yScale,near,far,nudge)
{
 var clipFactor=(far+near)/(far-near)
 return new Float32Array([
  xScale,0,0,0,
  0,yScale,0,0,
  0,0,-clipFactor,-1,
  0,0,-near*(clipFactor+1)+nudge,0
 ])
}


function drawSkyModel()
{
 if(skyboxModel.context!==context)
 {
  initSkyVBOs(skyboxModel);
 }
 context.bindBuffer(context.ARRAY_BUFFER,skyboxModel.aSkyCoord);
 context.vertexAttribPointer(aSkyCoord,3,context.FLOAT,false,0,0);
 context.drawArrays(context.TRIANGLES,0,skyboxModel.numCoordFloats/3);
}

function drawModel(model,position,alpha,spin)
{
 if(model.context!==context)
 {
  initVBOs(model);
 }
 context.bindBuffer(context.ARRAY_BUFFER,model.aCoord);
 context.vertexAttribPointer(aCoord,3,context.FLOAT,false,0,0);
 context.bindBuffer(context.ARRAY_BUFFER,model.aColor);
 context.vertexAttribPointer(aColor,3,context.FLOAT,false,0,0);
 context.uniform3fv(uObjectPosition,new Float32Array(position));
 context.uniformMatrix3fv(uObjectRotation,false,spin);
 context.uniform1f(uAlpha,alpha);
 context.drawArrays(context.TRIANGLES,0,model.numCoordFloats/3);
}

function drawDynamic(coords,colors,position,alpha,spin)
{
 context.bindBuffer(context.ARRAY_BUFFER,aCoordDynamic);
 context.bufferData(context.ARRAY_BUFFER,new Float32Array(coords),
		    context.DYNAMIC_DRAW);
 context.vertexAttribPointer(aCoord,3,context.FLOAT,false,0,0);
 context.bindBuffer(context.ARRAY_BUFFER,aColorDynamic);
 context.bufferData(context.ARRAY_BUFFER,new Float32Array(colors),
		    context.DYNAMIC_DRAW);
 context.vertexAttribPointer(aColor,3,context.FLOAT,false,0,0);
 context.uniform3fv(uObjectPosition,new Float32Array(position));
 context.uniformMatrix3fv(uObjectRotation,false,spin);
 context.uniform1f(uAlpha,alpha);
 context.drawArrays(context.TRIANGLES,0,coords.length/3);  
}

const VERTEX_SHADER_SOURCE=
      "uniform mat4 uPerspective;"+
      "uniform vec3 uCameraPosition;"+
      "uniform mat3 uCameraAngle;"+
      "uniform vec3 uObjectPosition;"+      
      "uniform mat3 uObjectRotation;"+
      "attribute vec3 aCoord;"+
      "attribute vec3 aColor;"+
      "varying lowp vec3 vColor;"+
      "varying highp vec3 vRelativeCoord;"+      
      "void main(void) {"+
      " vRelativeCoord=uObjectRotation*aCoord+uObjectPosition-uCameraPosition;"+
      " gl_Position=uPerspective*vec4(uCameraAngle*vRelativeCoord,1.0);"+
      " vColor=aColor;"+
      "}";

const FRAGMENT_SHADER_SOURCE=
      "varying lowp vec3 vColor;"+
      "uniform lowp float uAlpha;"+
      "varying highp vec3 vRelativeCoord;"+            
      "void main(void) {"+
      "gl_FragColor=vec4(vColor,uAlpha);"+
      "}";

const SKY_VERTEX_SHADER_SOURCE=
      "uniform mat4 uSkyPerspective;"+
      "uniform mat3 uSkyCameraAngle;"+
      "attribute vec3 aSkyCoord;"+
      "varying highp vec3 vSkyCoord;"+
      "void main(void) {"+
      " gl_Position=uSkyPerspective*vec4(uSkyCameraAngle*aSkyCoord,1.0);"+
      " vSkyCoord=aSkyCoord;"+
      "}";

const SKY_FRAGMENT_SHADER_SOURCE=
      "varying highp vec3 vSkyCoord;"+
      "void main(void) {"+
      " highp float theta=atan(vSkyCoord.y,length(vSkyCoord.xz));"+
      " highp float blend=theta+0.5;"+
      " blend=clamp(blend,0.0,1.0);"+
      " gl_FragColor=vec4(0.0,blend/16.0,blend/4.0,1.0);"+      
      "}";

function tryMove(mover,dimension,amount)
{
 if(amount<0) { return tryMinusMove(mover,dimension,-amount); }
 else { return tryPlusMove(mover,dimension,amount); }
}

function tryPlusMove(mover,dimension,amount)
{
 var initial=mover.position[dimension];
 mover.position[dimension]=initial+amount;
 var size=(dimension==1)?mover.height:mover.width;
 if(isInWall(mover))
 {
  // possibly dangerous assumption to remember:
  // mover.size is exactly representable
  mover.position[dimension]=Math.ceil(initial+size)-size;
  return false;
 }
 return true;
}

function tryMinusMove(mover,dimension,amount)
{
 var initial=mover.position[dimension];
 mover.position[dimension]=initial-amount;
 //if(dimension==1) { console.log(mover.position[1],amount); }
 if(isInWall(mover))
 {
  mover.position[dimension]=Math.floor(initial);
  return false;
 }
 return true;
}

function isInWall(mover)
{
 var right=mover.position[0]+mover.width;
 var maxX=Math.floor(right);
 if(right==maxX) --maxX;

 var south=mover.position[2]+mover.width;
 var maxZ=Math.floor(south);
 if(south==maxZ) --maxZ; 
 
 for(var x=Math.floor(mover.position[0]);
     x<=maxX;
     ++x)
 {
  for(var z=Math.floor(mover.position[2]);
      z<=maxZ;
      ++z)
  {
   if(editMode || z>=0 || z<worldHeightmap.length)
   {
    var row=worldHeightmap[z];
    var onMap=(row && x>=0 && x<row.length);
    if(editMode)
    {
     var y=onMap?row[x]:-1;
     if(y>mover.position[1]) { return true; }
    }
    else if(onMap)
    {
     var y=row[x];
     if(y>-1 && y>mover.position[1]) { return true; }
    }
   }
  }
 }
 return false;
}

function computeShadowCoords()
{
 var coords=[]

 var editCursor=(editMode && mouseHeld[2] && testIfInEditRange())
 
 var west=editCursor?getEditX()+.5-PLAYER_WIDTH/2:player.position[0]
 var east=west+PLAYER_WIDTH
 var maybeX=Math.floor(east);
 var north=editCursor?getEditZ()+.5-PLAYER_WIDTH/2:player.position[2]
 var south=north+PLAYER_WIDTH
 var maybeZ=Math.floor(south);
 var xCoords, zCoords;
 if(maybeX>west && maybeX<east) { xCoords=[west,maybeX,east]; }
 else { xCoords=[west,east]; }
 if(maybeZ>north && maybeZ<south) { zCoords=[north,maybeZ,south]; }
 else { zCoords=[north,south]; }
 

 for(var iz=0;iz<zCoords.length-1;++iz)
 {
  north=zCoords[iz];
  south=zCoords[iz+1];
  var row=worldHeightmap[Math.floor(north)];
  if(row || editCursor)
  {
   for(var ix=0;ix<xCoords.length-1;++ix)
   {
    west=xCoords[ix];
    east=xCoords[ix+1];
    var y;
    var px=Math.floor(west); 
    if(!row || px<0 || px>=row.length) { y=-1; }
    else { y=row[px]; }
    if((editCursor || y>=0) && y<player.position[1])
    {
     coords.push(west);
     coords.push(y);
     coords.push(north);
     
     coords.push(west);
     coords.push(y);
     coords.push(south);
     
     coords.push(east);
     coords.push(y);
     coords.push(north);
     
     coords.push(east);
     coords.push(y);
     coords.push(south);
     
     coords.push(east);
     coords.push(y);
     coords.push(north);
     
     coords.push(west);
     coords.push(y);
     coords.push(south);
    }
   }
  }
 }
 return coords
}

var keysHeld, mouseHeld,mouseAccumulator, keysFresh;

function onKeyDown(e)
{
 if(!e.shiftKey && !e.ctrlKey) { e.preventDefault(); }
 if(!keysHeld[e.code])
 {
  keysFresh[e.code]=true;
 }  
 keysHeld[e.code]=true;
}

function onKeyUp(e)
{
 if(!e.shiftKey && !e.ctrlKey) { e.preventDefault(); }
 keysHeld[e.code]=false;
}

function onPointerLockChange(e)
{
 if(document.pointerLockElement==canvas || editMode)
 {
  document.getElementById("pause_overlay").style.display="none";
 }
 else
 {
  document.getElementById("pause_overlay").style.display="block";  
 }
}


function onMouseDown(e)
{
 if(document.hasFocus() &&
    document.pointerLockElement!=canvas && !e.shiftKey && !e.ctrlKey)
 {
  canvas.requestPointerLock();
  mouseHeld={};
  mouseAccumulator=[0,0];
  keysFresh={}
 }
 else
 {
  // don't count the hold of the same click that's getting focus
  mouseHeld[e.button]=true;
 }
 e.preventDefault();
}

function onMouseUp(e)
{
 mouseHeld[e.button]=false;
 e.preventDefault(); 
}

function onMouseMove(e)
{
 if(document.pointerLockElement!=canvas) { return; }
 if(canvas.clientWidth && canvas.clientHeight)
 {
  mouseAccumulator[0]+=e.movementX/canvas.clientWidth;
  mouseAccumulator[1]+=e.movementY/canvas.clientHeight;
 }
 e.preventDefault();
}

const PLAYER_SPEED=1/32;
const PLAYER_POSITIVE_ACCEL=7/8192;
const PLAYER_NEGATIVE_ACCEL=16/8192;
const PLAYER_GRAVITY=4/4096;
const PLAYER_JUMP_FORCE=1/16;
const PLAYER_EDIT_HOVER=3/64;
const PLAYER_JUMP_CANCEL_THRESHOLD=1/32;

function getEditX()
{
 return Math.floor(player.position[0]+PLAYER_WIDTH/2);
}

function getEditZ()
{
 return Math.floor(player.position[2]+PLAYER_WIDTH/2);
}

function getFacingString()
{
 var f=Math.floor((cameraTheta*4/Math.PI)%8+8.5)%8;
 return getFacingNumber()+"\u00b0"+["N","NE","E","SE","S","SW","W","NW"][f];
}

function getSpawnFacingString()
{
 var f=Math.floor((spawnFacingNumber*4/180)%8+8.5)%8;
 return spawnFacingNumber+"\u00b0"+["N","NE","E","SE","S","SW","W","NW"][f];
}

function getFacingNumber()
{
 var f=Math.floor((cameraTheta*180/Math.PI)%360+360.5)%360;
 return f;
}

function respawnPlayer()
{
 player.xv=0;
 player.yv=0;
 player.zv=0;
 if(currentCheckpoint)
 {
  player.position=[Math.floor(currentCheckpoint.position[0])+.5-PLAYER_WIDTH/2,
		   Math.floor(currentCheckpoint.position[1]),
		   Math.floor(currentCheckpoint.position[2])+.5-PLAYER_WIDTH/2];
  cameraTheta=checkpointFacing;
 }
 else
 {
  player.position=spawnPosition.slice();
  cameraTheta=spawnFacingNumber*Math.PI/180;
 }
 cameraRawPhi=0.5; 
}

function resetMap()
{
 items=JSON.parse(itemsForReset);
 frameNumber=0;
 frameNumberTicking=false;
 itemsGotten=0;
 currentCheckpoint=null
 document.getElementById("buttonR").disabled=true;
 respawnPlayer();
}

const MAX_MAP_SIZE=256;


function gameLogicFrameCheck()
{
 var now=performance.now()
 if(document.pointerLockElement!=canvas)
 {
  gameLogicZeroOffset=now-1000/240*frameNumber;
  return;
 }
 var framesToRun=Math.floor((now-gameLogicZeroOffset)*240/1000-frameNumber);
 for(var i=0;i<framesToRun && i<10;++i)
 {
  gameLogicFrame();
 }
 if(framesToRun>10)
 {
  gameLogicZeroOffset=now-1000/240*frameNumber;
 }
}

function gameLogicFrame()
{
 if(frameNumberTicking)
 {
  frameNumber+=1;
 }
 
 if(editMode)
 {
  var oldEditX=getEditX(), oldEditZ=getEditZ();
  var oldFacingString=getFacingString();  
 }

 
 if(keysFresh["KeyR"] || overlayCommand=="R")
 {
  respawnPlayer()
 }
 if(overlayCommand=="M")
 {
  resetMap();
 }
 if(overlayCommand=="S")
 {
  currentCheckpoint=null;
  document.getElementById("buttonR").disabled=true;  
  respawnPlayer();
 }
 
 var dx=(keysHeld["KeyD"]?1:0)-(keysHeld["KeyA"]?1:0)
 var dz=(keysHeld["KeyS"]?1:0)-(keysHeld["KeyW"]?1:0) 

 if(dx && dz) { dx*=Math.sqrt(.5); dz*=Math.sqrt(.5); }

 if(dx||dz||mouseHeld[0]) { frameNumberTicking=true; }
 
 var cos=Math.cos(cameraTheta), sin=Math.sin(cameraTheta);
 
 var desiredXV=PLAYER_SPEED*(dx*cos-dz*sin);
 var desiredZV=PLAYER_SPEED*(dz*cos+dx*sin);

 var xa=desiredXV-player.xv;
 var za=desiredZV-player.zv;
 
 var dot=xa*player.xv+za*player.zv;
 var accel=(dot<0)?PLAYER_NEGATIVE_ACCEL:PLAYER_POSITIVE_ACCEL;
 

 var absa=Math.sqrt(xa*xa+za*za);
 if(absa>accel)
 {
  xa*=accel/absa;
  za*=accel/absa;
 }

 player.xv+=xa;
 player.zv+=za;

 if(player.yv<PLAYER_JUMP_CANCEL_THRESHOLD && player.yv>0 &&
    !mouseHeld[0]) { player.yv=0; }
 

 // split gravity between x and z move, to avoid
 // obvious asymmetry when navigating around weird corners
 
 player.yv-=PLAYER_GRAVITY/2;
 if(!tryMove(player,0,player.xv)) { player.xv=0; }
 var hitFloor=!tryMove(player,1,player.yv/2);
 if(hitFloor)
 {
  player.yv=mouseHeld[0]?PLAYER_JUMP_FORCE:0;
 }

 player.yv-=PLAYER_GRAVITY/2; 
 if(!tryMove(player,2,player.zv)) { player.zv=0; }
 hitFloor=!tryMove(player,1,player.yv/2);
 if(hitFloor)
 {
  player.yv=mouseHeld[0]?PLAYER_JUMP_FORCE:0;
 }

 if(player.position[1]<=-2 && !editMode) { respawnPlayer(); }
 
 var dTheta=mouseAccumulator[0];
 var dPhi=mouseAccumulator[1];

 cameraRawPhi+=dPhi*CAMERA_PHI_RATIO;
 if(cameraRawPhi<0) { cameraRawPhi=0; }
 if(cameraRawPhi>1) { cameraRawPhi=1; }
  
 cameraTheta+=dTheta*CAMERA_THETA_RATIO;
 cameraTheta%=Math.PI*2;

 
 var gotItemNow=false;
 for(var i=0;i<items.length;++i)
 {
  var it=items[i];
  if(player.position[0]+PLAYER_WIDTH>it.position[0] &&
     player.position[1]+PLAYER_HEIGHT>it.position[1] &&
     player.position[2]+PLAYER_WIDTH>it.position[2] &&
     player.position[0]<it.position[0]+ITEM_SIZE &&     
     player.position[1]<it.position[1]+ITEM_SIZE &&
     player.position[2]<it.position[2]+ITEM_SIZE &&
     !editMode)
  {
   it.garbage=true;
   itemsGotten++;
   gotItemNow=true;
  }
 }
 for(var i=0;i<checkpoints.length;++i)
 {
  var ch=checkpoints[i];
  if(currentCheckpoint!=ch &&
     player.position[0]+PLAYER_WIDTH>ch.position[0] &&
     player.position[1]+PLAYER_HEIGHT>ch.position[1] &&
     player.position[2]+PLAYER_WIDTH>ch.position[2] &&
     player.position[0]<ch.position[0]+CHECKPOINT_SIZE &&     
     player.position[1]<ch.position[1]+CHECKPOINT_SIZE &&
     player.position[2]<ch.position[2]+CHECKPOINT_SIZE &&
     !editMode)
  {
   checkpointFrame=frameNumber;
   currentCheckpoint=ch;
   document.getElementById("buttonR").disabled=false;   
   checkpointFacing=cameraTheta;
   gotItemNow=true;
  }
 }
 
 if(items.length)
 {
  items=items.filter(function(i) { return !i.garbage; });
  if(!items.length) { victoryFrame=frameNumber; }
 }

 if(frameNumber%240==0 || gotItemNow ||
    (currentCheckpoint && frameNumber==checkpointFrame+CHECKPOINT_DISPLAY_TIME))
 {
  updateText();
 }

 if(editMode)
 {
  if(mouseHeld[2] && player.yv<PLAYER_EDIT_HOVER)
  {
   player.yv=PLAYER_EDIT_HOVER;
  }
  if(player.position[1]>11) { player.position[1]=11; }
  var editX=getEditX();
  var editZ=getEditZ();

  var onMap=(editX>=0 && editZ>=0 &&
	     editZ<worldHeightmap.length &&
	     editX<worldHeightmap[editZ].length);

  var inEditRange=testIfInEditRange();

  function tryRaiseTile()
  {
   if(inEditRange &&
      (!onMap || worldHeightmap[editZ][editX]<9))
   {
    if(!onMap)
    {
     //console.log(JSON.stringify(worldHeightmap));
     expandWorldFor(editX,editZ);
     //console.log(JSON.stringify(worldHeightmap));    
     if(editX<0) { editX=0; }
     if(editZ<0) { editZ=0; }
    }
    
    worldHeightmap[editZ][editX]+=1;
    if(isInWall(player)) { player.position[1]+=1; }
    for(var i=0;i<items.length;++i)
    {
     if(items[i].blockX==editX &&
	items[i].blockZ==editZ)
     {
      parameterizeItem(worldHeightmap,items[i])
    }
    }
    for(var i=0;i<checkpoints.length;++i)
    {
     if(checkpoints[i].blockX==editX &&
	checkpoints[i].blockZ==editZ)
     {
      parameterizeCheckpoint(worldHeightmap,checkpoints[i])
     }
    }   
    if(Math.floor(spawnPosition[0])==editX &&
       Math.floor(spawnPosition[2])==editZ)
    {
     spawnPosition[1]+=1;
    }   
    refreshModelFromHeightmap(worldModel,worldHeightmap);
    history.replaceState("","","#"+encodeWorld())
    updateText();
    return true;
   }
   else { return false; }
  }

  function tryLowerTile()
  {
   if(canLowerTileAt(editX,editZ))     
   {
    worldHeightmap[editZ][editX]-=1;
    for(var i=0;i<items.length;++i)
    {
     if(items[i].blockX==editX &&
	items[i].blockZ==editZ)
     {
      parameterizeItem(worldHeightmap,items[i])
     }
    }
    for(var i=0;i<checkpoints.length;++i)
    {
     if(checkpoints[i].blockX==editX &&
	checkpoints[i].blockZ==editZ)
    {
     parameterizeCheckpoint(worldHeightmap,checkpoints[i])
    }
    }   
    if(Math.floor(spawnPosition[0])==editX &&
       Math.floor(spawnPosition[2])==editZ)
    {
     spawnPosition[1]-=1;
    }
    if(worldHeightmap[editZ][editX]==-1 &&
       ((editX==0 || editX==worldHeightmap[editZ].length-1) ||
	(editZ==0 || editZ==worldHeightmap.length-1)))
    {
     maybeContractWorld();
    }
    refreshModelFromHeightmap(worldModel,worldHeightmap);
    history.replaceState("","","#"+encodeWorld())
    updateText();
    return true;
   }
   return false;
  }

  function trySetTile(n)
  {
   var y;
   if(editZ<0 || editZ>=worldHeightmap.length) { y=-1; }
   else
   {
    var row=worldHeightmap[editZ];
    if(editX<0 || editX>=row.length) { y=-1; }
    else { y=row[editX]; }
   }
   var width=worldHeightmap[0].length;
   var depth=worldHeightmap.length;
   while(y>n && tryLowerTile() &&
	 width==worldHeightmap[0].length &&
	 depth==worldHeightmap.length)
   {y=worldHeightmap[editZ][editX];}
   while(y<n && tryRaiseTile()) {y=worldHeightmap[editZ][editX];}
  }
  
  if(keysFresh["KeyE"]) { tryRaiseTile(); }
  if(keysFresh["KeyQ"]) { tryLowerTile(); }
  if(keysFresh["KeyP"])
  {
   if(isTileClearForPit(editX,editZ))
   {
    trySetTile(-1);
   }
  }
  if(keysFresh["Digit1"] || keysFresh["Numpad1"]) { trySetTile(0); }
  if(keysFresh["Digit2"] || keysFresh["Numpad2"]) { trySetTile(1); }
  if(keysFresh["Digit3"] || keysFresh["Numpad3"]) { trySetTile(2); }
  if(keysFresh["Digit4"] || keysFresh["Numpad4"]) { trySetTile(3); }
  if(keysFresh["Digit5"] || keysFresh["Numpad5"]) { trySetTile(4); }
  if(keysFresh["Digit6"] || keysFresh["Numpad6"]) { trySetTile(5); }
  if(keysFresh["Digit7"] || keysFresh["Numpad7"]) { trySetTile(6); }
  if(keysFresh["Digit8"] || keysFresh["Numpad8"]) { trySetTile(7); }
  if(keysFresh["Digit9"] || keysFresh["Numpad9"]) { trySetTile(8); }
  if(keysFresh["Digit0"] || keysFresh["Numpad0"]) { trySetTile(9); }
  
  
  if(keysFresh["KeyZ"] && onMap &&
     worldHeightmap[editZ][editX]>-1)
  {
   tryRemoveItemAt(editX,editZ);
   tryRemoveCheckpointAt(editX,editZ);   
   spawnPosition[0]=editX+.5-PLAYER_WIDTH/2;
   spawnPosition[1]=worldHeightmap[editZ][editX];
   spawnPosition[2]=editZ+.5-PLAYER_WIDTH/2;
   spawnFacingNumber=getFacingNumber();
   respawnPlayer();
   history.replaceState("","","#"+encodeWorld())
   updateText();   
  }
  if(keysFresh["Space"] && onMap &&
     worldHeightmap[editZ][editX]>-1 &&
     (editX!=Math.floor(spawnPosition[0]) ||
      editZ!=Math.floor(spawnPosition[2])))
  {
   tryRemoveCheckpointAt(editX,editZ);
   var removed=tryRemoveItemAt(editX,editZ);
   if(!removed)
   {    
    var item={blockX:editX,blockZ:editZ}
    parameterizeItem(worldHeightmap,item);
    items.push(item);
   }
   updateText();
   history.replaceState("","","#"+encodeWorld())
  }
  if(keysFresh["KeyC"] && onMap &&
     worldHeightmap[editZ][editX]>-1 &&
     (editX!=Math.floor(spawnPosition[0]) ||
      editZ!=Math.floor(spawnPosition[2])))
  {
   tryRemoveItemAt(editX,editZ);
   var removed=tryRemoveCheckpointAt(editX,editZ);
   if(!removed)
   {    
    var checkpoint={blockX:editX,blockZ:editZ}
    parameterizeCheckpoint(worldHeightmap,checkpoint);
    checkpoints.push(checkpoint);
   }
   updateText();
   history.replaceState("","","#"+encodeWorld())
  }
  if(keysFresh["KeyJ"])
  {
   player.position[0]+=sin*10;
   player.position[2]-=cos*10;
   player.position[1]=11
  }
  if(editX!=oldEditX || editZ!=oldEditZ || getFacingString()!=oldFacingString)
  {
   updateText();
  }
 } 
 mouseAccumulator=[0,0];
 overlayCommand=null;
 keysFresh={}
}

function testIfInEditRange()
{
 var xNewSize, zNewSize;
 var editX=getEditX(), editZ=getEditZ();
 if(editX<0) { xNewSize=worldHeightmap[0].length-editX; }
 else if(editX>=worldHeightmap[0].length) { xNewSize=editX+1; }
 else { xNewSize=worldHeightmap[0].length; }
 if(editZ<0) { zNewSize=worldHeightmap.length-editZ; }
 else if(editZ>=worldHeightmap.length) { zNewSize=editZ+1; }   
 else { zNewSize=worldHeightmap.length; }
 return xNewSize<=MAX_MAP_SIZE && zNewSize<=MAX_MAP_SIZE;
}

function canLowerTileAt(x,z)
{
 if(x<0 || z<0 || z>=worldHeightmap.length || x>=worldHeightmap[z].length)
 {
  return false; // OOB is already pit
 }
 if(worldHeightmap[z][x]>0) { return true; } // still ground if lowered
 if(worldHeightmap[z][x]<0) { return false; } // already pit
 
 // lowering would make a pit: need to check if this pits an item or spawn
 return isTileClearForPit(x,z);
}

function isTileClearForPit(x,z)
{
 if(x==Math.floor(spawnPosition[0]) &&
    z==Math.floor(spawnPosition[2]))
 {
  return false;
 }
 for(var i=0;i<items.length;++i)
 {
  if(items[i].blockX==x &&
     items[i].blockZ==z)
  {
   return false;
  }
 }
 for(var i=0;i<checkpoints.length;++i)
 {
  if(checkpoints[i].blockX==x &&
     checkpoints[i].blockZ==z)
  {
   return false;
  }
 } 
 return true; 
}

function tryRemoveItemAt(x,z)
{
 var removed=false;
 for(var i=0;i<items.length;++i)
 {
  if(items[i].blockX==x &&
     items[i].blockZ==z)
  {
   items[i].garbage=true;
   removed=true;
  }
  items=items.filter(function(i) { return !i.garbage; });
 }
 return removed;
}


function tryRemoveCheckpointAt(x,z)
{
 var removed=false;
 for(var i=0;i<checkpoints.length;++i)
 {
  if(checkpoints[i].blockX==x &&
     checkpoints[i].blockZ==z)
  {
   checkpoints[i].garbage=true;
   removed=true;
  }
  checkpoints=checkpoints.filter(function(i) { return !i.garbage; });
 }
 return removed;
}

function expandWorldFor(newX,newZ)
{
 if(newX<0)
 {
  // add -newX to the start of every heightmap row
  for(var z=0;z<worldHeightmap.length;++z)
  {
   for(var i=0;i<-newX;++i)
   {
    worldHeightmap[z].splice(0,0,-1);
   }
  }
 }
 if(newX>=worldHeightmap[0].length)
 {
  // add newX+1-worldHeightmap[0].length to the end of every heightmap row
  var oldWidth=worldHeightmap[0].length;
  for(var z=0;z<worldHeightmap.length;++z)
  {
   for(var i=0;i<newX+1-oldWidth;++i)
   {
    worldHeightmap[z].push(-1); 
   }
  }
 }
 if(newZ<0)
 {
  // add -newZ heightmap rows at the start
  for(var i=0;i<-newZ;++i)
  {
   var newRow=[]
   for(var x=0;x<worldHeightmap[0].length;++x)
   {
    newRow.push(-1);
   }
   worldHeightmap.splice(0,0,newRow);
  }
 }
 if(newZ>=worldHeightmap.length)
 {
  // add newZ+1-worldHeightmap.length rows at the end
  var oldDepth=worldHeightmap.length;
  for(var i=0;i<newZ+1-oldDepth;++i)
  {
   var newRow=[]
   for(var x=0;x<worldHeightmap[0].length;++x)
   {
    newRow.push(-1);
   }
   worldHeightmap.push(newRow);
  }
 }
 if(newX<0 || newZ<0)
 {
  var xShift=newX<0?-newX:0;
  var zShift=newZ<0?-newZ:0;
  performItemAndParityShift(xShift,zShift);
 }
}


function maybeContractWorld()
{
 var xShift=0, zShift=0;
 while(isRectangleAllPit(worldHeightmap[0].length-1,0,
			 1,worldHeightmap.length))
 {
  for(var z=0;z<worldHeightmap.length;++z)
  {
   worldHeightmap[z].splice(worldHeightmap[z].length-1,1);
  }
 }
 while(isRectangleAllPit(0,worldHeightmap.length-1,
			 worldHeightmap[0].length,1)) 
 {
  worldHeightmap.splice(worldHeightmap.length-1,1);
 }
 while(isRectangleAllPit(0,0,
			 1,worldHeightmap.length))
 {
  xShift-=1;
  for(var z=0;z<worldHeightmap.length;++z)
  {
   worldHeightmap[z].splice(0,1);
  }
 }
 while(isRectangleAllPit(0,0,
			 worldHeightmap[0].length,1)) 
 {
  zShift-=1;
  worldHeightmap.splice(0,1);
 }
 if(xShift||zShift)
 {
  performItemAndParityShift(xShift,zShift);
 }
}

function performItemAndParityShift(xShift,zShift)
{
 for(var i=0;i<items.length;++i)
 {
  items[i].blockX+=xShift;
  items[i].blockZ+=zShift;
  parameterizeItem(worldHeightmap,items[i]);
 }
 for(var i=0;i<checkpoints.length;++i)
 {
  checkpoints[i].blockX+=xShift;
  checkpoints[i].blockZ+=zShift;
  parameterizeCheckpoint(worldHeightmap,checkpoints[i]);
 } 
 spawnPosition[0]+=xShift;
 spawnPosition[2]+=zShift;
 player.position[0]+=xShift;
 player.position[2]+=zShift;

 if((xShift&1) != (zShift&1)) { worldColorParity=1-worldColorParity; }   
}
				  

function isRectangleAllPit(west,north,width,depth)
{
 for(var z=north;z<north+depth;++z)
 {
  var row=worldHeightmap[z];
  for(var x=west;x<west+width;++x)
  {
   if(x>=0 && x<row.length && row[x]>-1) { return false; } 
  }
 }
 return true;
}


function timeString(f)
{
 var seconds=f/240;
 var minutes=Math.floor(seconds/60);
 seconds=Math.floor(seconds%60);
 return minutes+":"+(seconds<10?"0":"")+seconds;
}

const CHECKPOINT_DISPLAY_TIME=240;

function updateText()
{
 if(editMode)
 {
  document.getElementById("bottomtext").style["font-size"]="2vh";
  document.getElementById("bottomtext").innerHTML=(
   "Game controls are WASD, R, mouse movement, and the left mouse button.<br>"+
   "Hold right mouse button to hover and to show a cursor shadow.<br>"+
    "Press E and Q to raise and lower blocks, Space to place and remove RAINBOW BULLSHIT, and Z to set the spawn point.<br>"+
    "1-9, 0, and P quick-set block height. J teleports forward. C places and removes checkpoints. "+  
    "Save the page URL to share and play your map."
  );
  document.getElementById("toptext").style["font-size"]="3vh";  
  document.getElementById("toptext").innerHTML=(
   "("+(getEditX()-Math.floor(spawnPosition[0]))+
    ", "+(getEditZ()-Math.floor(spawnPosition[2]))+
    ") from spawn point, current facing "+getFacingString()+"<br>"+
    " world size "+(worldHeightmap[0].length)+"x"+
    (worldHeightmap.length)+" (max "+MAX_MAP_SIZE+"x"+MAX_MAP_SIZE+")"+
    "<br>"+
    "spawn point facing "+getSpawnFacingString()+", "+
    items.length+" RAINBOW BULLSHIT placed"
  );
  return;
 }
 
 if(items.length || itemsGotten==0)
 {
  document.getElementById("bottomtext").style["font-size"]="7vh"
  document.getElementById("bottomtext").innerHTML=items.length+" RAINBOW BULLSHIT REMAINING";  
  document.getElementById("toptext").style["font-size"]="7vh"      
  document.getElementById("toptext").innerHTML=timeString(frameNumber);  
 }
 else
 {
  document.getElementById("bottomtext").style["font-size"]="7vh"  
  document.getElementById("bottomtext").innerHTML="ALL RAINBOW BULLSHIT COLLECTED!";
  document.getElementById("toptext").style["font-size"]="3vh"    
  document.getElementById("toptext").innerHTML=
   "MAP CLEARED IN "+timeString(victoryFrame)+"<br>"+
   "Pause and click RESET MAP to play again.";  
 }
 if(itemsGotten==0)
 {
  document.getElementById("bottomtext").style["font-size"]="3vh"
  document.getElementById("bottomtext").innerHTML=
   "Game controls are WASD, R, mouse movement, and the left mouse button.<br>"+
   "Collect all the RAINBOW BULLSHIT!";
 }
 if(currentCheckpoint && frameNumber-checkpointFrame<CHECKPOINT_DISPLAY_TIME)
 {
  document.getElementById("bottomtext").style["font-size"]="7vh"    
  document.getElementById("bottomtext").innerHTML="CHECKPOINT SET!";
 }
}

function parameterizeItem(heightmap,item)
{
 var x=item.blockX;
 var z=item.blockZ;
 var y=heightmap[z][x];
 item.position=[x+.5-ITEM_SIZE/2,
		 y+.5-ITEM_SIZE/2,
		z+.5-ITEM_SIZE/2];
 item.drawPosition=[x+.5,
		    y+.5,
		    z+.5];
 item.width=ITEM_SIZE;
 item.height=ITEM_SIZE;
}

function parameterizeCheckpoint(heightmap,checkpoint)
{
 var x=checkpoint.blockX;
 var z=checkpoint.blockZ;
 var y=heightmap[z][x];
 checkpoint.position=[x+.5-CHECKPOINT_SIZE/2,
		 y+.5-CHECKPOINT_SIZE/2,
		z+.5-CHECKPOINT_SIZE/2];
 checkpoint.drawPosition=[x+.5,
		    y+.5,
		    z+.5];
 checkpoint.width=CHECKPOINT_SIZE;
 checkpoint.height=CHECKPOINT_SIZE;
}



function encodeHeightmap(heightmap)
{
 var bits=[]
 var runLength=0, runY;

 function writeRun(length,y)
 {
  // this gets called on run of 25 or shorter.
  if(length>0)
  {
   bits.push(y);
   if(length>1)
   {
    // length 1: implicit because not followed by a letter
    // length 2-26: "b" to "z". "b" == 96+2
    // length 27-52: "A" to "Z". "A" == 38+27
    if(length<27) { bits.push(String.fromCharCode(96+length)); }
    else { bits.push(String.fromCharCode(38+length)); }
   }
  }
 }
 
 function endRun()
 {
  while(runLength>52)
  {
   writeRun(52,runY);
   runLength-=52;
  }
  writeRun(runLength,runY);
  runLength=0;
 }
 
 for(var z=0;z<heightmap.length;++z)
 {
  for(var x=0;x<heightmap[0].length;++x)
  {
   var y=heightmap[z][x]|0;
   if(y==-1) { y="~"; }
   if(runLength==0) { runY=y; runLength=1; }
   else if(runY!=y) { endRun(); runLength=1; runY=y; }
   else { runLength++; }
  }
 }
 endRun()
 return bits.join("");
}

function encodeStuff()
{
 var bits=[
 ]
 bits.push(spawnFacingNumber);
 bits.push(Math.floor(spawnPosition[0]));
 bits.push(Math.floor(spawnPosition[2]));
 for(var i=0;i<items.length;++i)
 {
  bits.push(items[i].blockX);
  bits.push(items[i].blockZ); 
 }
 return bits.join(",");
}

function encodeCheckpoints()
{
 var bits=[
 ]
 for(var i=0;i<checkpoints.length;++i)
 {
  bits.push(checkpoints[i].blockX);
  bits.push(checkpoints[i].blockZ); 
 }
 return bits.join(","); 
}

function encodeWorld()
{
 return (worldHeightmap[0].length+
	 (worldColorParity?"X":"x")+
	 worldHeightmap.length+"."+
	 encodeHeightmap(worldHeightmap)+"!"+
	 encodeStuff()+"!"+
	 encodeCheckpoints()
	);
}

function decodeWorld(str)
{
 if(!str) { return false; }
 var matched=str.match(/(\d+)([xX])(\d+).([0-9A-Za-z~]+)!([0-9,]+)!([0-9,]*)/);
 if(!matched) { return false; }
 var trailingNumberPart=matched[5].split(",");
 if(trailingNumberPart.length<3 || trailingNumberPart.length%2!=1)
 {
  return false;
 }
 var secondTrailingNumberPart=matched[6].split(",");
 if(secondTrailingNumberPart[0]=="")
 {
  secondTrailingNumberPart.splice(0,1);
 }
 if(secondTrailingNumberPart.length%2==1)
 {
  return false;
 }
 var xSize=matched[1]|0, zSize=matched[3]|0;
 if(xSize<1 || zSize<1 || xSize>999 || zSize>999) { return false; }
 var heightmap=decodeHeightmap(xSize,zSize,matched[4]);
 if(!heightmap) { return false; }
 var spawnFacing=(trailingNumberPart[0]|0)%360;
 var spawnX=trailingNumberPart[1]|0
 var spawnZ=trailingNumberPart[2]|0;
 if(spawnX<0 || spawnZ<0 || spawnX>=xSize || spawnZ>=zSize) { return false; }
 spawnPosition=[spawnX+.5-PLAYER_WIDTH/2,
		heightmap[spawnZ][spawnX],
		spawnZ+.5-PLAYER_WIDTH/2];
 var items=[]
 for(var i=3;i<trailingNumberPart.length;i+=2)
 {
  var item={blockX:trailingNumberPart[i]|0,
	    blockZ:trailingNumberPart[i+1]|0};
  if(item.blockX<0 || item.blockX>=xSize ||
     item.blockZ<0 || item.blockZ>=zSize) { return false; }
  parameterizeItem(heightmap,item);
  items.push(item);
 }
 var checkpoints=[]
 for(var i=0;i<secondTrailingNumberPart.length;i+=2)
 {
  var checkpoint={blockX:secondTrailingNumberPart[i]|0,
		  blockZ:secondTrailingNumberPart[i+1]|0};
  if(checkpoint.blockX<0 || checkpoint.blockX>=xSize ||
     checkpoint.blockZ<0 || checkpoint.blockZ>=zSize) { return false; }
  parameterizeCheckpoint(heightmap,checkpoint);
  checkpoints.push(checkpoint);
 }

 
 return {heightmap:heightmap,
	 spawnPosition:spawnPosition,
	 spawnFacing:spawnFacing,
	 items:items,
	 colorParity:matched[2]=="X"?1:0,
	 checkpoints:checkpoints
	};
}

function decodeHeightmap(xSize,zSize,str)
{
 var ret=new Array(zSize)
 for(var i=0;i<zSize;++i) { ret[i]=new Array(xSize); }
 var unpackedPosition=0;
 var packedPosition=0;
 while(packedPosition<str.length)
 {
  var y;
  if(str[packedPosition]=="~") { y=-1; }
  else { y=str[packedPosition]|0; }
  if(y<-1 || y>9) { return false; }
  ++packedPosition;
  var following=str[packedPosition];
  var runLength;
  if(following>="a" && following<="z")
  {
   runLength=following.charCodeAt(0)-96;
   ++packedPosition;
  }
  else if(following>="A" && following<="Z")
  {
   runLength=following.charCodeAt(0)-38;
   ++packedPosition;
  }
  else
  {
   runLength=1;
  }
  while(runLength>0)
  {
   if(unpackedPosition>=zSize*xSize) { return false; }
   ret[Math.floor(unpackedPosition/xSize)][unpackedPosition%xSize]=y;
   --runLength;
   ++unpackedPosition;
  }
 }
 if(unpackedPosition<zSize*xSize) { return false; }
 return ret;
}


	   
function makeEmptyHeightmap(xSize,zSize,floor)
{
 var ret=new Array(zSize);
 for(var z=0;z<zSize;++z)
 {
  var row=new Array(xSize);
  for(var x=0;x<xSize;++x)
  {
   row[x]=floor;
  }
  ret[z]=row;
 }
 return ret;
}

function tameButton(e)
{
 e.stopPropagation();
}

function overlayClick(c)
{
 overlayCommand=c;
 canvas.requestPointerLock();
 mouseHeld={};
 mouseAccumulator=[0,0];
 keysFresh={}
}
		   
